package gov.va.vamf.scheduling.direct.eligibilitychecker;

import com.agilex.healthcare.directscheduling.dataservice.CancelCodes;
import com.agilex.healthcare.directscheduling.dataservice.DSFacilityProviderDataService;
import com.agilex.healthcare.directscheduling.domain.BookedAppointment;
import com.agilex.healthcare.directscheduling.domain.BookedAppointments;
import com.agilex.healthcare.directscheduling.domain.PatientVisited;
import com.agilex.healthcare.directscheduling.utils.DateHelper;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import gov.va.vamf.scheduling.direct.datalayer.appointment.AppointmentDataService;
import gov.va.vamf.scheduling.direct.datalayer.clinic.ClinicDataService;
import gov.va.vamf.scheduling.direct.datalayer.eligibility.DirectBookingEligibilityCriteriaService;
import gov.va.vamf.scheduling.direct.datalayer.eligibility.RequestEligibilityCriteriaService;
import gov.va.vamf.scheduling.direct.domain.CdwClinic;
import gov.va.vamf.scheduling.direct.domain.CdwClinics;
import gov.va.vamf.scheduling.direct.domain.CoreSetting;
import gov.va.vamf.scheduling.direct.domain.DirectBookingEligibilityCriteria;
import gov.va.vamf.scheduling.direct.domain.RequestEligibilityCriteria;
import gov.va.vamf.scheduling.direct.domain.StopCode;
import gov.va.vamf.scheduling.direct.domain.StopCodes;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.HashMap;
import java.util.function.Function;

@Component
public class EligibilityCheckWorker {
    public static final int MONTHS_IN_YEAR = 12;
    public static final int ONE_YEAR_IN_DAYS = 365;
    public static final String DIRECT_ELIGIBILITY_TYPE = "direct";
    public static final String REQUEST_ELIGIBILITY_TYPE = "request";

    private final ClinicDataService clinicDataService;
    private final AppointmentDataService appointmentDataService;
    private final RequestEligibilityCriteriaService requestEligibilityCriteriaService;
    private final DSFacilityProviderDataService dsFacilityProviderDataService;
    private final DirectBookingEligibilityCriteriaService directBookingEligibilityCriteriaService;

    @Autowired
    public EligibilityCheckWorker(
            final ClinicDataService clinicDataService,
            final AppointmentDataService appointmentDataService,
            final RequestEligibilityCriteriaService requestEligibilityCriteriaService,
            final DSFacilityProviderDataService dsFacilityProviderDataService,
            final DirectBookingEligibilityCriteriaService directBookingEligibilityCriteriaService
    ) {
        this.clinicDataService = clinicDataService;
        this.appointmentDataService = appointmentDataService;
        this.requestEligibilityCriteriaService = requestEligibilityCriteriaService;
        this.dsFacilityProviderDataService = dsFacilityProviderDataService;
        this.directBookingEligibilityCriteriaService = directBookingEligibilityCriteriaService;
    }

    public BookedAppointments retrievePastDaysBookedAppointments(PatientIdentifier patientIdentifier, String siteCode, int days) {
        Date startDate = DateHelper.minusDays(new Date(), days);

        return appointmentDataService.retrieveBookedAppointments(patientIdentifier, siteCode, startDate);
    }

    private boolean appointmentClinicMatchTypeOfCareStopCodes(String clinicId, String siteCode, StopCodes stopCodes) {

        CdwClinics clinics = clinicDataService.retrieveClinicsByStopCodes(siteCode, stopCodes);

        for (CdwClinic clinic : clinics) {
            if (clinic.getClinicId().equals(clinicId) && clinic.getSiteCode().equals(siteCode)) {
                StopCode clinicStopCode = new StopCode(clinic.getPrimaryStopCode(), clinic.getSecondaryStopCode());

                for (StopCode typeOfCareStopCode : stopCodes) {
                    if (StopCode.match(typeOfCareStopCode, clinicStopCode)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    public PatientVisited hasVisitedClinicInPastMonthsByTypeOfCare(PatientIdentifier patientIdentifier, String siteCode, String clinicalServiceId, String institutionCode, String eligibilityType) {

        Boolean hasVisited = false;
        int historyDurationInDays;

        switch(eligibilityType) {
            case DIRECT_ELIGIBILITY_TYPE:
                historyDurationInDays = getDirectHistoryDurationInDays(clinicalServiceId, institutionCode);
                break;

            case REQUEST_ELIGIBILITY_TYPE:
                historyDurationInDays = getRequestHistoryDurationInDays(clinicalServiceId, institutionCode);
                break;

            default:
                historyDurationInDays = 0;
        }

        if (historyDurationInDays != 0) {
            StopCodes stopCodes = dsFacilityProviderDataService.createStopCodesFromPrimary(clinicalServiceId);

            Date beginDate = DateHelper.minusDays(new Date(), historyDurationInDays);

            HashMap<String, Boolean> clinicMatchedMap = new HashMap<>();
            Boolean clinicMatched;

            BookedAppointments bookedAppointments = retrievePastDaysBookedAppointments(patientIdentifier, siteCode, historyDurationInDays);

            for (final BookedAppointment bookedAppointment : bookedAppointments) {
                if (!hasValidBookedAppointment(bookedAppointment)) {
                    continue;
                }

                String clinicId = bookedAppointment.getClinic().getId();

                clinicMatched = clinicMatchedMap.get(clinicId);

                if (clinicMatched == null) {
                    clinicMatched = appointmentClinicMatchTypeOfCareStopCodes(clinicId, siteCode, stopCodes);
                }

                if (hasVisits(clinicMatched, bookedAppointment, beginDate)) {

                    hasVisited = true;

                    break;
                }

                clinicMatchedMap.put(clinicId, clinicMatched);
            }
        }

        PatientVisited patientVisited = new PatientVisited();
        patientVisited.setDurationInMonths(historyDurationInDays != 0 ? historyDurationInDays / ONE_YEAR_IN_DAYS * MONTHS_IN_YEAR : 0);
        patientVisited.setHasVisitedInPastMonths(hasVisited);

        return patientVisited;
    }

    public int getDirectHistoryDurationInDays(String clinicalServiceId, String institutionCode) {
        int durationInDays = 0;

        final DirectBookingEligibilityCriteria directBookingEligibilityCriteria =
                directBookingEligibilityCriteriaService.retrieveDirectBookingEligibilityCriteria(institutionCode);

        if (directBookingEligibilityCriteria != null) {
            durationInDays = directBookingEligibilityCriteria.getSetting(clinicalServiceId)
                    .map(new Function<CoreSetting, Integer>() { // TODO: switch to lambda
                        @Override
                        public Integer apply(CoreSetting coreSetting) {
                            return coreSetting.getPatientHistoryDuration();
                        }
                    })
                    .orElse(0);
        }

        return durationInDays;
    }

    public int getRequestHistoryDurationInDays(String clinicalServiceId, String institutionCode) {
        int durationInDays = 0;

        final RequestEligibilityCriteria requestEligibilityCriteria = requestEligibilityCriteriaService.retrieveRequestEligibilityCriteria(institutionCode);

        if (requestEligibilityCriteria != null) {
            durationInDays = requestEligibilityCriteria.getSetting(clinicalServiceId)
                    .map(new Function<CoreSetting, Integer>() { // TODO: switch to lambda
                        @Override
                        public Integer apply(CoreSetting coreSetting) {
                            return coreSetting.getPatientHistoryDuration();
                        }
                    })
                    .orElse(0);
        }

        return durationInDays;
    }

    public Boolean hasVisits(Boolean clinicMatched, BookedAppointment bookedAppointment, Date beginDate) {
        if (clinicMatched) {

            String currentStatus = bookedAppointment.getCurrentStatus();

            if (currentStatus != null &&
                    (currentStatus.contains("CHECKED IN") || currentStatus.contains("CHECKED OUT"))) {

                if (bookedAppointment.getAppointmentTime().after(beginDate)) {
                    return true;
                }
            }
        }

        return false;
    }

    public Boolean hasValidBookedAppointment(BookedAppointment bookedAppointment) {
        if (bookedAppointment.getCurrentStatus() != null && !CancelCodes.hasCancelCode(bookedAppointment.getCurrentStatus())) {

            if (bookedAppointment.getClinic() != null && !bookedAppointment.getClinic().getId().isEmpty()) {
                return true;
            }
        }

        return false;
    }
}
